package com.tongda.tdtopo.service;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.tongda.tdtopo.neo4j.Neo4jServer;
import com.tongda.tdtopo.util.GraphUtil;
import lombok.Data;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.neo4j.graphdb.*;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;

import java.io.ByteArrayInputStream;
import java.nio.charset.StandardCharsets;
import java.util.*;

@Service
@Component
@Data
public class TDGraphService {
    //测试
    private  GraphDatabaseService graphDb=new Neo4jServer().getGraphDatabaseServiceIn();

    public static HashMap<String ,String> keymap=new HashMap<>();
    static {
        keymap.put("version","integer");
        keymap.put("valueMax","double");
        keymap.put("valueMin","double");
    }

    /**
     * 添加元数据
     * 新xml格式输入
     * 重复的内容会覆盖
     * @param xml 元数据扁平化格式结果
     * @return  成功返回 true
     */
    public boolean addMetaData(String xml){
        //清理数据
        //MATCH (n)-[r]-() WITH r,n LIMIT 1000000 DETACH DELETE r,n RETURN count(*);

        System.out.println("接收数据开始处理:"+xml);
        long totalMilliSeconds1 = System.currentTimeMillis();
        try(Transaction transaction = graphDb.beginTx()) {


            //解析XML数据
            SAXReader saxReader = new SAXReader();
            Document document = saxReader.read(new ByteArrayInputStream(xml.getBytes(StandardCharsets.UTF_8)));
            Element graph = document.getRootElement();
            Element reNodes=graph.element("Nodes");
            //处理节点
            List<Element> elements=reNodes.elements("Node");
            long totalMilliSeconds2 = System.currentTimeMillis();
//            System.out.println("xml处理时间："+(totalMilliSeconds2-totalMilliSeconds1));

            Map<String ,Node> nodeMap=new HashMap<>();

            //节点处理循环
            for (Element e : elements) {
                //判断节点是否存在

                long totalMilliSecondsSelect1 = System.currentTimeMillis();
                Node node = graphDb.findNode(Label.label(e.attributeValue("tag")), "id", e.attributeValue("id")==null?"null":e.attributeValue("id"));
                long totalMilliSecondsSelect2 = System.currentTimeMillis();
//                System.out.println("查询节点耗时："+(totalMilliSecondsSelect2-totalMilliSecondsSelect1));
                if (node == null) {
                    //不存在创建空节点
                    long totalMilliSeconds = System.currentTimeMillis();
                    node=graphDb.createNode();
                    long totalMilliSeconds3 = System.currentTimeMillis();
//                    System.out.println("库操作/创建空节点耗时："+(totalMilliSeconds3-totalMilliSeconds));
                } else {
                    //存在则循环移除属性
                    long totalMilliSeconds = System.currentTimeMillis();
                    Iterable<String> i=node.getPropertyKeys();
                    for (String s:i) {
                        node.removeProperty(s);
                    }
                    long totalMilliSeconds4 = System.currentTimeMillis();
//                    System.out.println("库操作/重置属性耗时："+(totalMilliSeconds4-totalMilliSeconds));
                }

                //设置标签
                node.addLabel(Label.label(e.attributeValue("tag")));

                //判断dataType属性
                long totalMilliSeconds5 = System.currentTimeMillis();
                if (e.attributeValue("dataType") != null) {
                    String[] dataTypes = e.attributeValue("dataType").split("\\.");
                    System.out.println(e.attributeValue("dataType"));
                    System.out.println(Arrays.toString(dataTypes));
                    String dataType=dataTypes[dataTypes.length-1];
                    node.setProperty("dataType",dataType);
                    if(e.attributeValue("mode")!=null){
                        if (e.attributeValue("mode").endsWith(".multi.select.value")||e.attributeValue("mode").endsWith("range.value")){
                            dataType="string";
                        }
                    }


                    //设置value
                    if (e.attributeValue("value")!=null&&(!e.attributeValue("value").equals(""))){
                        switch (dataType) {
                            case "string": {
                                node.setProperty("value",e.attributeValue("value"));
                                e.remove(e.attribute("dataType"));
                                e.remove(e.attribute("value"));
                                break;
                            }
                            case "integer": {
                                int value = Integer.parseInt(e.attributeValue("value"));
                                node.setProperty("value",value);
                                System.out.println("-----------------------------------------");
                                e.remove(e.attribute("dataType"));
                                e.remove(e.attribute("value"));
                                break;
                            }
                            case "float": {
                                float value = Float.parseFloat(e.attributeValue("value"));
                                node.setProperty("value",value);
                                e.remove(e.attribute("dataType"));
                                e.remove(e.attribute("value"));
                                break;
                            }
                        }
                    }

                }
                long totalMilliSeconds6 = System.currentTimeMillis();
//                System.out.println("库操作/处理value："+(totalMilliSeconds6-totalMilliSeconds5));

                //处理特定字段的值类型，字段类型集合keymap
                for (Attribute a : e.attributes()) {
                    if (keymap.get(a.getName()) == null) {
                        node.setProperty(a.getName(),a.getValue());
                    } else {
                        String type = keymap.get(a.getName());
                        switch (type) {
                            case "long":
                                node.setProperty(a.getName(),Long.valueOf(a.getValue()));
                                break;
                            case "integer":
                                node.setProperty(a.getName(),Integer.valueOf(a.getValue()) );
                                break;
                            case "double":
                                node.setProperty(a.getName(),Double.valueOf(a.getValue()));
                                break;
                        }
                    }

                }
                long totalMilliSeconds7 = System.currentTimeMillis();
//                System.out.println("库操作/处理属性字段类型总耗时："+(totalMilliSeconds7-totalMilliSeconds6));
                //将节点对象添加到map
                nodeMap.put(e.attributeValue("id"),node);
            }

            long totalMilliSeconds8 = System.currentTimeMillis();
//            System.out.println("库操作/节点处理总耗时:"+(totalMilliSeconds8-totalMilliSeconds2));

            //先删除中央节点向下的关系
            long totalMilliSeconds9= System.currentTimeMillis();
            String id = graph.attributeValue("rootID");
            String tag="";
            for (Element e : elements) {
                if(id.equals(e.attributeValue("id"))){
                    tag=e.attributeValue("tag");
                }
            }
            //删除线测试
            removeGraphRelation(id);
//            Node rootNode = graphDb.findNode(Label.label(tag), "id",id);
//            Iterable<Relationship> resourceIterable=rootNode.getRelationships();
//            for (Relationship relationship:resourceIterable){
//                String s=relationship.getType().name();
//                if (s.equals("HAS_METADATA")) {
//                    relationship.delete();
////                        System.out.println("执行删除");
//                }
//
//            }
            long totalMilliSeconds10 = System.currentTimeMillis();
//            System.out.println("库操作/删除中央节点向下的关系耗时："+(totalMilliSeconds10-totalMilliSeconds9));

            //重新添加关系
            List<Element> relationships = graph.element("Edges").elements("Edge");
            for (Element e : relationships) {
                Node fromNode=nodeMap.get(e.attributeValue("from"));
                Node toNode=nodeMap.get(e.attributeValue("to"));
                if(fromNode==null){
                    String fromTag=e.attributeValue("from").split("-")[0];
                    fromNode = graphDb.findNode(Label.label(fromTag), "id", e.attributeValue("from"));

                }
                if(toNode==null){
                    String toTag=e.attributeValue("to").split("-")[0];
                    toNode = graphDb.findNode(Label.label(toTag), "id", e.attributeValue("to"));
                }

                if (fromNode!=null&&toNode!=null){
                    RelationshipType relationshipType=RelationshipType.withName(e.attributeValue("relType"));
                    long totalMilliSeconds11 = System.currentTimeMillis();
                    Iterable<Relationship> iterator=fromNode.getRelationships();
                    Relationship relationship=null;
                    for (Relationship relationship1:iterator){
                        if(relationship1.getProperty("hash").equals(e.attributeValue("hash"))){
                            relationship=relationship1;
                        }
                    }
                    long totalMilliSeconds12 = System.currentTimeMillis();
//                    System.out.println("库操作/查询关系耗时："+(totalMilliSeconds12-totalMilliSeconds11));
                    if (relationship==null){
                        relationship=fromNode.createRelationshipTo(toNode,relationshipType);
                        for (Attribute a : e.attributes()) {
                            relationship.setProperty(a.getName(),a.getValue());
                        }
                    }
                }else {
                    System.out.println("连线节点不存在："+fromNode+"/"+toNode);
                }
            }
            long totalMilliSeconds11 = System.currentTimeMillis();
//            System.out.println("库操作/添加关系总耗时："+(totalMilliSeconds11-totalMilliSeconds10));
            long totalMilliSeconds12 = System.currentTimeMillis();
            transaction.success();
            long totalMilliSeconds13 = System.currentTimeMillis();
            System.out.println("success耗时:"+(totalMilliSeconds13-totalMilliSeconds12));
            System.out.println("提交------------------------------------------------------------------------------");
        } catch (DocumentException e) {
//            System.out.println(e.toString());
            throw new RuntimeException(e);
        }
        return true;
    }

    /**
     * 只更新节点node
     * @param xml
     * @return
     */
    public String updateNodes(String xml){

        try(Transaction transaction = graphDb.beginTx()) {
            //解析XML数据
            SAXReader saxReader = new SAXReader();
            Document document = saxReader.read(new ByteArrayInputStream(xml.getBytes(StandardCharsets.UTF_8)));
            Element root = document.getRootElement();
            //处理节点
            List<Element> elements=root.elements("Nodes");
            Element nodes=elements.get(0);
            for (Element e : nodes.elements()) {
                //属性拼装对象
                String s = "";
                if (e.attributeValue("dataType") != null) {
                    String dataType = e.attributeValue("dataType");
                    switch (dataType) {
                        case "string": {
                            String value = e.attributeValue("value");
                            s = s + "value:" + value + ",";
                            e.remove(e.attribute("value"));
                            break;
                        }
                        case "integer": {
                            int value = Integer.parseInt(e.attributeValue("value"));
                            s = s + "value:" + value + ",";
                            e.remove(e.attribute("value"));
                            break;
                        }
                        case "float": {
                            float value = Float.parseFloat(e.attributeValue("value"));
                            s = s + "value:" + value + ",";
                            e.remove(e.attribute("value"));
                            break;
                        }
                    }
                }

                //处理特定字段的值类型，字段类型集合keymap
                for (Attribute a : e.attributes()) {
                    if (keymap.get(a.getName()) == null) {
                        s = s + a.getName() + ":'" + a.getValue() + "',";
                    } else {
                        String type = keymap.get(a.getName());
                        switch (type) {
                            case "long":
                                s = s + a.getName() + ":" + Long.valueOf(a.getValue()) + ",";
                                break;
                            case "integer":
                                s = s + a.getName() + ":" + Integer.valueOf(a.getValue()) + ",";
                                break;
                            case "double":
                                s = s + a.getName() + ":" + Double.valueOf(a.getValue()) + ",";
                                break;
                        }
                    }
                }

                s = s.substring(0, s.length() - 1);
                //根据id查询节点是否存在，不存在新增，存在则将属性全部重新set一遍
                Node node = graphDb.findNode(Label.label(e.attributeValue("tag")), "id", e.attributeValue("id"));
                if (node != null){
                    String reset = "MATCH (n:" + e.attributeValue("tag") + ") WHERE n.id = '" + e.attributeValue("id") +
                            "' SET n = {" + s + "}";
                    graphDb.execute(reset);
                }
            }
            transaction.success();
        } catch (DocumentException e) {
            throw new RuntimeException(e);
        }

        return "true";
    }

    /**
     *移除节点
     * @param xml
     * @return
     */
    public String removeNodes(String xml){

        try(Transaction transaction = graphDb.beginTx()) {
            //解析XML数据
            SAXReader saxReader = new SAXReader();
            Document document = saxReader.read(new ByteArrayInputStream(xml.getBytes(StandardCharsets.UTF_8)));
            Element root = document.getRootElement();

            //重新添加关系
            List<Element> relationships = root.elements("Edges");
            Element relations = relationships.get(0);
            //删除关系
            for (Element e : relations.elements()) {
                String relationMerge ="match (n)-[r{from:'"+e.attributeValue("from")+"', to:'"+e.attributeValue("to")+"'}]-(m) delete  r";
                System.out.println(relationMerge);
                graphDb.execute(relationMerge);
            }

            //删除节点
            List<Element> elements=root.elements("Nodes");
            Element nodes=elements.get(0);
            for (Element e : nodes.elements()) {
                //根据id查询节点是否存在，不存在新增，存在则将属性全部重新set一遍
                String remove = "match (n{id:'"+e.attributeValue("id")+"'}) delete  n";
                graphDb.execute(remove);
            }

            transaction.success();
        } catch (DocumentException e) {
            throw new RuntimeException(e);
        }

        return "true";
    }

    /**
     * 根据id添加节点tag
     * @param json
     * {
     *     "tag": "TagTwo",
     *     "idList": [
     *         "Product-1112041739683627008",
     *         "MetadataEntry-1112041739683627015"
     *     ]
     * }
     * @return
     */
    public String addTag(String json){

        try(Transaction transaction = graphDb.beginTx()) {
            JSONObject jsonObject= JSON.parseObject(json);
            String  tag = jsonObject.get("tag").toString();
            List<String > idList= (List<String>) jsonObject.get("idList");

            String addTag="match (n) where ";
            for (String s: idList) {
                addTag=addTag+"n.id='"+s+"' or ";
            }
            addTag=addTag.substring(0,addTag.length()-3);
            addTag=addTag+" set n:"+ tag+" return n";
//        System.out.println(addTag);
            graphDb.execute(addTag);
            transaction.success();
        }


        return "true";
    }

    /**
     * 根据id删除节点
     * @param json
     * {
     *     "idList": [
     *         "CandidateValue-1094237139551387649",
     *         "MetadataEntry-1092842332425814016"
     *     ]
     * }
     * @return
     */
    public String removeNode(String json){

        try(Transaction transaction = graphDb.beginTx()) {
            JSONObject jsonObject= JSON.parseObject(json);
            List<String > idList= (List<String>) jsonObject.get("idList");
            String cypher="match (n) where ";
            for (String s: idList) {
                cypher=cypher+"n.id='"+s+"' or ";
            }
            cypher=cypher.substring(0,cypher.length()-3);
            cypher=cypher+" delete n";
            graphDb.execute(cypher);
            transaction.success();

        }

        return "true";
    }

    /**
     * 删除节点及关系
     * @param json
     * {
     *     "idList": [
     *         "CandidateValue-1094237139551387649",
     *         "MetadataEntry-1092842332425814016"
     *     ]
     * }
     * @return
     */
    public String removeNodeAndRelation(String json){

        try(Transaction transaction = graphDb.beginTx()){
            JSONObject jsonObject= JSON.parseObject(json);
            List<String > idList= (List<String>) jsonObject.get("idList");
            String cypher="match (n) where ";
            for (String s: idList) {
                cypher=cypher+"n.id='"+s+"' or ";
            }
            cypher=cypher.substring(0,cypher.length()-3);
            cypher=cypher+" DETACH DELETE n";

            graphDb.execute(cypher);
            transaction.success();
        }


        return "true";
    }

    /**
     * 根据id删除节点及以下可以删除的节点和关系
     */
    public String removeGraph(String id){
        String cypher="";
        String cypher2="";
        try(Transaction transaction = graphDb.beginTx()){
            String tag=id.split("-")[0];
            switch (tag){
                case "MDET":
                    cypher="match (n:"+tag+")-[r:HAS_VALUE]->(m) where n.id='"+id+"' DETACH DELETE n,m";
                    break;
                case "ASTT":
                    cypher="match (n:"+tag+")-[r:HAS_METADATA]->(m) where n.id='"+id+"' DETACH DELETE n,m";
                    break;
                case "AST":
                    cypher="match (n:"+tag+")-[r:HAS_METADATA]->(m) where n.id='"+id+"' DETACH DELETE n,m";
                    break;
                case "REST":
                    cypher="match (n:"+tag+")-[r:HAS_METADATA]->(m) where n.id='"+id+"' DETACH DELETE n,m";
                    break;
                case "RES":
                    cypher="match (n:"+tag+")-[r:HAS_METADATA]->(m) where n.id='"+id+"' DETACH DELETE n,m";
                    break;
                case "TEM":
                    cypher="match (n:"+tag+")-[r:HAS_ROLE]->(m) where n.id='"+id+"' DETACH DELETE n,m";
                    cypher2="match (m:Product)<-[t:HAS_ASSET]-(n:"+tag+"{id:'"+id+"'})-[r:HAS_ASSET]->(p:Product) " +
                            "match (m)-[y{depend:'"+id+"'}]->(p) " +
                            "delete y";
                    break;
            }

            Result result;
            if(!cypher2.equals("")){
                result=graphDb.execute(cypher2);
            }
            result=graphDb.execute(cypher);

//            System.out.println(result.toString());
            transaction.success();
        }

        return "true";
    }

    public String removeGraphRelation(String id){
        String cypher="";
        String cypher2="";
        try(Transaction transaction = graphDb.beginTx()){
            String tag=id.split("-")[0];
            switch (tag){
                case "MDET":
                    cypher="match (n:"+tag+")-[r:HAS_VALUE]->(m) where n.id='"+id+"' DETACH DELETE r";
                    break;
                case "ASTT":
                    cypher="match (n:"+tag+")-[r:HAS_METADATA]->(m) where n.id='"+id+"' DETACH DELETE r";
                    break;
                case "AST":
                    cypher="match (n:"+tag+")-[r:HAS_METADATA]->(m) where n.id='"+id+"' DETACH DELETE r";
                    break;
                case "REST":
                    cypher="match (n:"+tag+")-[r:HAS_METADATA]->(m) where n.id='"+id+"' DETACH DELETE r";
                    break;
                case "RES":
                    cypher="match (n:"+tag+")-[r:HAS_METADATA]->(m) where n.id='"+id+"' DETACH DELETE r";
                    break;
                case "TEM":
                    cypher="match (n:"+tag+")-[r:HAS_ROLE]->(m) where n.id='"+id+"' DETACH DELETE r";
                    cypher2="match (m:Product)<-[t:HAS_ASSET]-(n:"+tag+"{id:'"+id+"'})-[r:HAS_ASSET]->(p:Product) " +
                            "match (m)-[y{depend:'"+id+"'}]->(p) " +
                            "delete y";
                    break;
            }

            Result result;
            if(!cypher2.equals("")){
                result=graphDb.execute(cypher2);
            }
            result=graphDb.execute(cypher);

//            System.out.println(result.toString());
            transaction.success();
        }


        return "true";
    }

    /**
     * 移除关系
     * @param json
     * {
     *     "fromID": "aaa",
     *     "toID": "bbb",
     *     "relation": "HAS_RESOURCE"
     * }
     * @return
     */
    public String removeRelation(String json){
        //{”fromID“:"aaa","toID":"bbb","relation":"HAS_RESOURCE"}
        try(Transaction transaction = graphDb.beginTx()){
            JSONObject jsonObject= JSON.parseObject(json);
            String cypher="match (n{id:'"+jsonObject.get("fromID")+"'})-[r:"+jsonObject.get("relation")+"]->(m{id:'"+jsonObject.get("toID")+"'}) " +
                    "DELETE r";
            System.out.println(cypher);
            graphDb.execute(cypher);
            transaction.success();
        }


        return "true";
    }

    /**
     * 添加关系
     * @param json
     * {
     *     "fromID": "aaa",
     *     "toID": "bbb",
     *     "relation": "HAS_RESOURCE"
     * }
     * @return
     */
    public String addRelation(String json){
        //{”fromID“:"aaa","toID":"bbb","relation":"HAS_RESOURCE"}
        try(Transaction transaction = graphDb.beginTx();){
            JSONObject jsonObject= JSON.parseObject(json);
            String cypher="match (n{id:'"+jsonObject.get("fromID")+"'}),"+
                    "(m{id:'"+jsonObject.get("toID")+"'}) " +
                    "MERGE (n)-[r:"+jsonObject.get("relation")+"]->(m)";
            graphDb.execute(cypher);
            transaction.success();
        }


        return "true";
    }

    /**
     * 完全删除一个标签的标记
     * @param json
     * {
     *     "tag": "aaa"
     * }
     * @return
     */
    public String removeTagAllNode(String json){
        try(Transaction transaction = graphDb.beginTx()){
            JSONObject jsonObject= JSON.parseObject(json);
            String cypher="MATCH (n) REMOVE n:"+jsonObject.get("tag")+" ";
            graphDb.execute(cypher);
            transaction.success();
        }


        return "true";
    }

    /**
     * 根据id删除指定节点的标签
     * @param json
     * {
     *     "tag": "aaa",
     *     "id": "aaaa8"
     * }
     * @return
     */
    public String removeTagWithNode(String json){
        try(Transaction transaction = graphDb.beginTx()){
            JSONObject jsonObject= JSON.parseObject(json);
            String cypher="MATCH (n{id:'"+jsonObject.get("id")+"'}) REMOVE n:"+jsonObject.get("tag")+" ";
            graphDb.execute(cypher);
            transaction.success();
        }

        return "true";
    }

    /**
     * 根据tag和id查询节点，id是或的关系
     * @param json
     *{
     *     "tag": [
     *         "ProductTemplate"
     *     ],
     *     "idList": [
     *         "ProductTemplate-1112009760707706880"
     *     ]
     * }
     * @return
     */
    public String searchMetadata(String json){

        try(Transaction transaction = graphDb.beginTx()){
            JSONObject jsonObject= JSON.parseObject(json);
            List<String > tagList= (List<String>) jsonObject.get("tag");

            //添加tag条件
            String cypher="match (n";
            for (String s:
                    tagList) {
                cypher=cypher+":"+s;
            }
            cypher=cypher+") ";

            //添加id条件
            List<String > idList= (List<String>) jsonObject.get("idList");
            if (idList.size()!=0){
                cypher=cypher+"where ";
                for (String s:
                        idList) {
                    cypher=cypher+"n.id='"+s+"' or ";
                }
                cypher=cypher.substring(0,cypher.length()-3);
            }

            cypher=cypher+" return n";
            Result result=graphDb.execute(cypher);

            ArrayList<String> list=new ArrayList<>();
            while (result.hasNext()) {
                Map<String,Object> row = result.next();
                for (Map.Entry<String,Object> column : row.entrySet()) {
                    JSONObject json1 = new JSONObject(((Node)column.getValue()).getAllProperties());
                    list.add(json1.toString());
                }
            }

            transaction.success();
            return list.toString();
        }
    }

    /**
     * 根据tag和label查询节点,id是或的关系
     * @param json
     * {
     *     "tag": [
     *         "MetadataEntry"
     *     ],
     *     "label": "温度"
     * }
     * @return
     */
    public String searchByLabel(String json){
        try(Transaction transaction = graphDb.beginTx()){
            JSONObject jsonObject= JSON.parseObject(json);
            List<String > tagList= (List<String>) jsonObject.get("tag");

            //添加tag条件
            String cypher="match (n";
            for (String s:
                    tagList) {
                cypher=cypher+":"+s;
            }
            cypher=cypher+") ";
            //模糊查询内容
            cypher=cypher+" where n.label contains '"+jsonObject.get("label")+"' return n";
            System.out.println(cypher);
            Result result=graphDb.execute(cypher);

            ArrayList<String> list=new ArrayList<>();
            while (result.hasNext()) {
                Map<String,Object> row = result.next();
                for (Map.Entry<String,Object> column : row.entrySet()) {
                    JSONObject json1 = new JSONObject(((Node)column.getValue()).getAllProperties());
                    list.add(json1.toString());
                }
            }
            transaction.success();

            return list.toString();
        }



    }

    /**
     * 根据tag和vakue查询节点,id是或的关系
     * {
     *     "tag": [
     *         "MetadataEntry"
     *     ],
     *     "value": "温度"
     * }
     * @return
     */
    public List<String > searchByValueOrLabel(String[] tag,String queryData){

        try(Transaction transaction = graphDb.beginTx()){
            //添加tag条件
            System.out.println(Arrays.toString(tag));
            String cypher="match (m";
            for (String s:
                    tag) {
                cypher=cypher+":"+s;
            }
            cypher=cypher+") -[r*1]->(n) ";
            //模糊查询内容
            cypher=cypher+" where n.value contains '"+queryData+"'"+" or n.label contains '"+queryData+"'"+" return m";
            System.out.println(cypher);
            Result result=graphDb.execute(cypher);

            ArrayList<String> list=new ArrayList<>();
            while (result.hasNext()) {
                Map<String,Object> row = result.next();
                for (Map.Entry<String,Object> column : row.entrySet()) {
                    JSONObject json1 = new JSONObject(((Node)column.getValue()).getAllProperties());
                    list.add(json1.toString());
                }
            }
            transaction.success();

            return list;
        }
    }

    /**
     * 查询节点的关系
     * @param json
     *  {"id":"MetadataEntry-1094237139551387648"}
     * @return
     */
    public String nodeRelationType(String json){
        try(Transaction transaction = graphDb.beginTx()){
            JSONObject jsonObject= JSON.parseObject(json);
            String id= jsonObject.get("id").toString();
            String type=id.split("-")[0];
            String cypher="match " +
                    "(n)-[i]-(m:"+type+"{id:\""+id+"\"}) " +
                    "return DISTINCT type(i)";

            System.out.println(cypher);
            Result result=graphDb.execute(cypher);

            List<JSONObject> list=new ArrayList<>();

            while (result.hasNext()) {
                Map<String,Object> row = result.next();
                for (Map.Entry<String,Object> column : row.entrySet()) {
                    JSONObject jsonType=new JSONObject();
                    jsonType.put("type",column.getValue().toString());
//                    jsonType.put("label","");
                    list.add(jsonType);
                }
            }

            HashMap<String ,List<JSONObject>> map=new HashMap<>();
            map.put("up",list);
            transaction.success();

            return JSONObject.toJSONString(map);
        }
    }

    /**
     *根据 节点id，上下行，关系类别 查询节点
     * @param json
     * {
     *     "id": "ProductTemplate-1112009760707706880",
     *     "type": "HAS_METADATA",
     *     "directional": "down"
     * }
     * @return
     */
    public String nodeAndRelationToNode(String json){
        try(Transaction transaction = graphDb.beginTx()){
            JSONObject jsonObject= JSON.parseObject(json);
            String type=jsonObject.get("id").toString().split("-")[0];
            String cypher="match (n)-[i:"+jsonObject.get("type").toString()+"]-(m:"+type+"{id:\""+jsonObject.get("id").toString()+"\"}) return DISTINCT properties(n)";

            Result result=graphDb.execute(cypher);
            List<Object> listNode=new ArrayList<>();
            while (result.hasNext()) {
                Map<String, Object> row = result.next();
                for (Map.Entry<String, Object> column : row.entrySet()) {
                    listNode.add(column.getValue());
                }
            }
            transaction.success();
            return JSONObject.toJSONString(listNode);
        }
    }

    /**
     * 添加自定义标签字典表节点
     * CustomizationTag
     * @param json
     * {
     *     "name": "TagTwo",
     *     "label": "自定义标签",
     *     "description": "xxxx类型的标记"
     * }
     * @return
     */
    public String addTagType(String json){
        try(Transaction transaction = graphDb.beginTx()){
            JSONObject jsonObject= JSON.parseObject(json);
            String cypher="merge (n:CustomizationTag"+
                    "{name:'"+jsonObject.get("name")+"',label:'"+jsonObject.get("label")+"',description:'"+jsonObject.get("description")+"'})";
            graphDb.execute(cypher);
            transaction.success();

            return "true";
        }
    }

    /**
     * 查询自定义标签表
     * @return
     */
    public String searchTagType(){
        try(Transaction transaction = graphDb.beginTx()){
            String cypher="match (n:CustomizationTag) return n";
            Result result=graphDb.execute(cypher);
            ArrayList<String> list=new ArrayList<>();
            while (result.hasNext()) {
                Map<String,Object> row = result.next();
                for (Map.Entry<String,Object> column : row.entrySet()) {
                    JSONObject json1 = new JSONObject(((Node)column.getValue()).getAllProperties());
                    list.add(json1.toString());
                }
            }
            transaction.success();
            return list.toString();
        }

    }

    /**
     * 删除自定义标签
     * @param json
     * {"name":"TagTwo"}
     * @return
     */
    public String removeTagType(String json){
        try(Transaction transaction = graphDb.beginTx();){
            JSONObject jsonObject= JSON.parseObject(json);
            String cypher="match (n:CustomizationTag{name:'"+jsonObject.get("name")+"'}) delete n";
            graphDb.execute(cypher);
            transaction.success();


            return "true";
        }
    }

    /**
     * 添加自定义关系类型字典表节点
     * @param json
     * {
     *     "name": "TagTwo",
     *     "label": "自定义关系",
     *     "description": "xxxx类型的关系"
     * }
     * @return
     */
    public String addRelationType(String json){
        try(Transaction transaction = graphDb.beginTx()){
            JSONObject jsonObject= JSON.parseObject(json);
            String cypher="merge (n:CustomizationRelation"+
                    "{name:'"+jsonObject.get("name")+"',label:'"+jsonObject.get("label")+"',description:'"+jsonObject.get("description")+"'})";
            graphDb.execute(cypher);
            transaction.success();


            return "true";
        }
    }

    /**
     * 查询自定义关系类型表
     * @return
     */
    public String searchRelationType(){
        try(Transaction transaction = graphDb.beginTx()){
            String cypher="match (n:CustomizationRelation) return n";
            Result result=graphDb.execute(cypher);
            ArrayList<String> list=new ArrayList<>();
            while (result.hasNext()) {
                Map<String,Object> row = result.next();
                for (Map.Entry<String,Object> column : row.entrySet()) {
                    JSONObject json1 = new JSONObject(((Node)column.getValue()).getAllProperties());
                    list.add(json1.toString());
                }
            }
            transaction.success();

            return list.toString();
        }
    }

    /**
     * 删除自定义关系类型
     * {"name":"TagTwo"}
     * @param json
     * @return
     */
    public String  removeRelationType(String json){
        try(Transaction transaction = graphDb.beginTx()){
            JSONObject jsonObject= JSON.parseObject(json);
            String cypher="match (n:CustomizationRelation{name:'"+jsonObject.get("name")+"'}) delete n";
            graphDb.execute(cypher);
            transaction.success();

            return "true";
        }
    }

    /**
     * 获取一个节点下的所有信息
     *  //获取下级节点
     *     //  match (n{id:"Product-1092049602028240896aaa"})-[*1..5]->(m)
     *     //  return DISTINCT
     *     //  n,m
     *     //获取所有下级关系
     *     //match (n{id:"Product-1092049602028240896aaa"})-[r*1..5]->(m)
     *     // return DISTINCT r
     * @param json
     * @return
     */
    public String  searchGraph(String json){

        try(Transaction transaction = graphDb.beginTx()){
            JSONObject jsonObject= JSON.parseObject(json);

            //获取中心节点
            String tage=jsonObject.get("id").toString().split("-")[0];
            String cypher="match (n:"+tage+"{id:'"+jsonObject.get("id")+"'}) return DISTINCT n";
            Result result=graphDb.execute(cypher);
            HashMap<String,JSONObject> centerNodeMap= GraphUtil.nodesUtil(result);
            Iterator<String> iterator=centerNodeMap.keySet().iterator();
            String id="";
            JSONObject centerNode=null;
            if(iterator.hasNext()){
                id=iterator.next();
                centerNode=centerNodeMap.get(id);
            }

            //下行步长内的节点
            cypher="match (n:"+tage+"{id:'"+jsonObject.get("id")+"'})-[*1.."+jsonObject.get("downsteps")+"]->(m) " +
                    "return DISTINCT n,m";
            System.out.println(cypher);
            result=graphDb.execute(cypher);
            HashMap<String,JSONObject> hashMapNode=GraphUtil.nodesUtil(result);
            hashMapNode.remove(id);
            //下行步长内的关系
            cypher="match p=(n:"+tage+"{id:'"+jsonObject.get("id")+"'})-[r*1.."+jsonObject.get("downsteps")+"]->(m) return DISTINCT relationships(p)";
            System.out.println(cypher);
            result=graphDb.execute(cypher);
            ArrayList<JSONObject> relationList=GraphUtil.linksUtil(result);
            //map转list
            JSONObject returnJson=new JSONObject();
            ArrayList<JSONObject> nodeList=new ArrayList<>();
            for (String s:hashMapNode.keySet()){
                JSONObject jsonObject1=hashMapNode.get(s);
                nodeList.add(jsonObject1);
            }
            //上行步长内的节点
            cypher="match (n:"+tage+"{id:'"+jsonObject.get("id")+"'})<-[*1.."+jsonObject.get("upsteps")+"]-(m)" +
                    "return DISTINCT n,m";
            result=graphDb.execute(cypher);
            HashMap<String,JSONObject> parentNodes=GraphUtil.nodesUtil(result);
            parentNodes.remove(id);
            ArrayList<JSONObject> parentNodeList=new ArrayList<>();
            for (String s:parentNodes.keySet()){
                JSONObject jsonObject1=parentNodes.get(s);

                parentNodeList.add(jsonObject1);
            }
            //上行步长内的关系
            cypher="match p=(n:"+tage+"{id:'"+jsonObject.get("id")+"'})<-[r*1.."+jsonObject.get("upsteps")+"]-(m) return DISTINCT relationships(p)";
            result=graphDb.execute(cypher);
            ArrayList<JSONObject> parentRelationList=GraphUtil.linksUtil(result);

            nodeList.add(centerNode);
            nodeList.addAll(parentNodeList);
            returnJson.put("nodes",nodeList);
            relationList.addAll(parentRelationList);
            returnJson.put("links",relationList);

            String re=returnJson.toString();
            transaction.close();
            return re;
        }
    }

    /**
     *根据id查询图
     * @param json
     * [
     *     "Product-1112041472149946368",
     *     "Product-1112039689512026112"
     * ]
     * @return
     */
    public String  searchIdGraph(String json){
        try(Transaction transaction = graphDb.beginTx()){
            JSONArray jsonArray= JSONObject.parseObject(json,JSONArray.class);

            //id列表内的节点
            String cypher="match (n) where n.id in "+jsonArray.toString()+"  return DISTINCT n";
            System.out.println(cypher);
            Result result=graphDb.execute(cypher);
            HashMap<String,JSONObject> hashMapNode=GraphUtil.nodesUtil(result);

            //map转list
            JSONObject returnJson=new JSONObject();
            ArrayList<JSONObject> nodeList=new ArrayList<>();
            for (String s:hashMapNode.keySet()){
                JSONObject jsonObject1=hashMapNode.get(s);
                jsonObject1.put("id",s);
                nodeList.add(jsonObject1);
            }

            //id列表内的节点间的关系
            cypher="match (n)-[i*1]->(m) where n.id in "+jsonArray+" " +
                    "and" +
                    "" + " m.id in "+jsonArray+"   return DISTINCT i";
            System.out.println(cypher);
            result=graphDb.execute(cypher);
            ArrayList<JSONObject> relationList=GraphUtil.linksUtil(result);

            returnJson.put("nodes",nodeList);
            returnJson.put("links",relationList);
            String re=returnJson.toString();
            transaction.close();
            return re;
        }
    }

    /**
     *  数据统计的接口
     * match (n)-[HAS_METADATA]->(m) where (n.id="Product-1092049602028240896" or n.id="Product-1092049602028240896aaa")
     * and (m.key="fai.env.temperature" or m.key="fai.env.working.temperature"  or m.key="book.student.math") return n,m
     * @param json
     * {
     *     "idList": ["Product-1092049602028240896","Product-1092049602028240896aaa"],
     *     "tagList": ["Product"],
     *     "keyList": ["fai.env.temperature","fai.env.working.temperature","book.student.math"]*
     * }
     * @return
     */
    public String  searchData(String json) {
        try(Transaction transaction = graphDb.beginTx()){
            JSONObject inputJson= JSON.parseObject(json);
            List<String > tagList= (List<String>) inputJson.get("tagList");

            //添加tag
            String cypher="match (n";
            for (String s:
                    tagList) {
                cypher=cypher+":"+s;
            }
            cypher=cypher+")-[HAS_METADATA]->(m) ";

            //添加id条件
            List<String > idList= (List<String>) inputJson.get("idList");
            if (idList.size()!=0){
                cypher=cypher+"where (";
                for (String s:
                        idList) {
                    cypher=cypher+"n.id='"+s+"' or ";
                }
                cypher=cypher.substring(0,cypher.length()-4)+")";
            }
            //添加key
            cypher=cypher+" and (";
            List<String > keyList= (List<String>) inputJson.get("keyList");
            if (keyList.size()!=0){
                for (String s:
                        keyList) {
                    cypher=cypher+"m.key='"+s+"' or ";
                }
                cypher=cypher.substring(0,cypher.length()-4)+")";
            }

            cypher=cypher+" return n.id,m";
            Result result=graphDb.execute(cypher);
            JSONObject vector=new JSONObject();

            ArrayList<JSONObject> scalarList=new ArrayList<>();
            HashMap<String ,ArrayList<JSONObject>> hashMap=new HashMap<>();
            while (result.hasNext()) {
                Map<String,Object> row = result.next();
                Set<Map.Entry<String, Object>> set = row.entrySet();
                Iterator<Map.Entry<String, Object>> i = set.iterator();
                Map.Entry<String, Object> entry1=i.next();
                ArrayList<JSONObject> dataList;
                String key=entry1.getValue().toString();
                if (hashMap.get(key)==null){
                    dataList=new ArrayList<>();
                }else {
                    dataList=hashMap.get(entry1.getValue().toString());
                }
                Map.Entry<String, Object> entry2=i.next();
                JSONObject jsonEntry = new JSONObject(((Node)entry2.getValue()).getAllProperties());
                JSONObject data =new JSONObject();
                data.put("uom",jsonEntry.get("nom"));
                data.put("value",jsonEntry.get("value"));
                data.put("dataType",jsonEntry.get("dataType"));
                data.put("key",jsonEntry.get("key"));
                data.put("label",jsonEntry.get("label"));
                dataList.add(data);
                hashMap.put(key,dataList);
            }
            for (String s:
                    hashMap.keySet()) {
                JSONObject jsonObject=new JSONObject();
                jsonObject.put("scalar",hashMap.get(s));
                scalarList.add(jsonObject);

            }
            JSONObject inputDataSet=new JSONObject();
            vector.put("Vector",scalarList);
            inputDataSet.put("inputDataSet",vector);
            transaction.success();

            return inputDataSet.toString();
        }


    }

    /**
     * 查询全部数据（暂不使用）
     * @return
     */
    public String searchGraphAll() {

        try(Transaction transaction = graphDb.beginTx()) {
            String cypher = "match (n)-[r*1]-(m) return n,r,m limit 3000";
            System.out.println(cypher);
            Result result = graphDb.execute(cypher);

            JSONObject returnJson = GraphUtil.LinksAndNodesUtil(result);

            String re = returnJson.toString();
            transaction.close();
            return re;
        }


    }

    /**
     * 高级查询
     * {
     *     "tags": [
     *         "Product",
     *         "Resource"
     *     ],
     *     "label": "Oracle数据库",
     *     "keys": [
     *         {
     *             "key": "MetadataEntryTemplate-responsible_person_name",
     *             "value": "刘三"
     *         }
     *         ,
     *         {
     *             "key": "MetadataEntryTemplate-responsible_department",
     *             "value": "建设部"
     *         }
     *     ]
     * }
     * @param jsonObject
     * @return
     */
    public HashMap<String ,HashMap<String,List<String>>> searchAllByValueOrLabel(JSONObject jsonObject) {

        List<String> tags=JSONArray.parseArray(jsonObject.get("tags").toString(),String.class);
        String label=jsonObject.getString("label");
        List<HashMap> keys=JSONArray.parseArray(jsonObject.get("keys").toString(),HashMap.class);

        try(Transaction transaction = graphDb.beginTx()){
            //添加tag条件
            String cypher="match (n) where (";
            for (String s:
                    tags) {
                cypher=cypher+" n:"+s+" or";
            }
            cypher = cypher.substring(0, cypher.length() - 2);
            cypher=cypher+") and n.label contains '"+label+"' with n as p ";

            if (keys.size()>0){
                cypher=cypher+"match ";
                int index=0;
                String where=" where ";
                for (Map map:keys){
                    cypher=cypher+"(p)-->(p"+index+"{key:'"+map.get("key")+"'}),";
                    String datatype=map.get("datatype").toString();
                    String symbol=map.get("symbol").toString();
                    if(datatype.equals("number")){
                        switch (symbol) {
                            case "=" :
                                where=where+" p"+index+".value="+map.get("value")+" and";
                                break;
                            case ">" :
                                where=where+" p"+index+".value>"+map.get("value")+" and";
                                break;
                            case "<" :
                                where=where+" p"+index+".value<"+map.get("value")+" and";
                                break;
                        }
                    }
                    if(datatype.equals("string")){
                        switch (symbol) {
                            case "=" :
                                where=where+" p"+index+".value='"+map.get("value")+"' and";
                                break;
                            case "contains" :
                                where=where+" p"+index+".value contains '"+map.get("value")+"' and";
                                break;
                        }
                    }
                    index++;
                }
                cypher = cypher.substring(0, cypher.length() - 1);
                where = where.substring(0, where.length() - 3);
                cypher=cypher+where;
            }
            cypher=cypher+" return p";
            System.out.println(cypher);
            Result result=graphDb.execute(cypher);
            ArrayList<String> list=new ArrayList<>();
            HashMap<String ,HashMap<String,List<String>>> hashMap=new HashMap();
            while (result.hasNext()) {
                Map<String,Object> row = result.next();
                for (Map.Entry<String,Object> column : row.entrySet()) {
                    JSONObject jsonNode = new JSONObject(((Node)column.getValue()).getAllProperties());
//                    System.out.println("node:"+jsonNode);
                    String tag="";
                    if (jsonNode.get("tag")==null){
                        tag="default";
                    }else {
                        tag=jsonNode.get("tag").toString();
                    }
                    HashMap<String,List<String>> tagMap=hashMap.get(tag);
                    String template=jsonNode.get("template").toString();
                    if (tagMap==null){
                        tagMap=new HashMap<>();
                    }
                    List<String> nodelist=tagMap.get(template);
                    if(nodelist==null){
                        nodelist=new ArrayList<>();
                    }
                    nodelist.add(jsonNode.toString());
                    tagMap.put(template,nodelist);
                    hashMap.put(tag,tagMap);

                }
            }

            return hashMap;
        }
    }

    /**
     *  查询tag类型
     * @return
     */
    public List<String>  searchTagsType() {
        try(Transaction transaction = graphDb.beginTx()){
            String cypher="match(n) return distinct labels(n)";
            Result result=graphDb.execute(cypher);
            List<String > tagsList=new ArrayList<>();
            while (result.hasNext()) {
                Map<String,Object> row = result.next();
                for (Map.Entry<String,Object> column : row.entrySet()) {
                    ArrayList<String> arrayList= (ArrayList<String>) column.getValue();
                    if (arrayList.size()==1){
                        tagsList.add(arrayList.get(0));
                    }
                }
            }
            transaction.success();
            return tagsList;
        }
    }

    /**
     *  删除孤立的节点
     * @return
     */
    public void  deleteLoneNode() {
        try(Transaction transaction = graphDb.beginTx()){
            String cypher="match(n) where not(n) -- () delete(n)";
            Result result=graphDb.execute(cypher);
            transaction.success();
        }
    }

}
