package com.yingxin.prms.api;

import com.yingxin.prms.config.TopologyConfig;
import com.yingxin.prms.service.business.Manager;
import com.yingxin.prms.service.dao.CassandraDaoImpl;
import jdk.nashorn.internal.runtime.JSONListAdapter;
import net.sf.json.JSON;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.AsyncTaskExecutor;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.Map;


@RestController
@RequestMapping("/rest/rel")
@CrossOrigin(origins = "*")
public class ReliabilityApi {

    private final Manager manager;
    private final AsyncTaskExecutor taskExecutor;
    private final CassandraDaoImpl cassandraDao;
    private final TopologyConfig topologyConfig;

    @Autowired
    public ReliabilityApi(Manager manager, AsyncTaskExecutor taskExecutor, CassandraDaoImpl cassandraDao, TopologyConfig topologyConfig) {
        this.manager = manager;
        this.taskExecutor = taskExecutor;
        this.cassandraDao = cassandraDao;
        this.topologyConfig = topologyConfig;
    }

    /**
     * 2. 可靠性信息接口：所有集群的状态
     *
     * @return 集群信息json字符串
     */
    @GetMapping("selectClusterStatus")
    public Map selectClusterStatus() {
        return manager.selectClusterStatus();
    }

    @GetMapping("getThreadPoolStatus")
    public Map<String, Object> getThreadPoolStus() {
        ThreadPoolTaskExecutor executor = (ThreadPoolTaskExecutor)taskExecutor;
        int activeCount = executor.getActiveCount();
        int corePoolSize = executor.getCorePoolSize();
        int keepAliveSeconds = executor.getKeepAliveSeconds();
        int maxPoolSize = executor.getMaxPoolSize();
        int poolSize = executor.getPoolSize();
        Map<String, Object> map = new HashMap<>();
        map.put("activeCount", activeCount);
        map.put("corePoolSize", corePoolSize);
        map.put("keepAliveSeconds", keepAliveSeconds);
        map.put("maxPoolSize", maxPoolSize);
        map.put("poolSize", poolSize);
        return map;
    }

    @PostMapping("getAlarmMsg")
    public Map<String, Object> getAlarmMsg() {
        return manager.getAlarmMsg();
    }
    @PostMapping("searchLog")
    public Map<String, Object> searchLog(@RequestBody String json) {
        return manager.searchLog(json);
    }

    @PostMapping("login")
    public Map<String, Object> login(@RequestBody String json) {
        return manager.login(json);
    }

    @PostMapping("getToken")
    public Map<String, Object> getToken(@RequestBody String json) {
        return manager.getToken(json);
    }


    /**
     * 获取折线图
     *
     * @return 集群信息json字符串
     */
    @PostMapping("selectChart")
    public Map selectChart(@RequestBody String json) throws Exception{
        return manager.selectChart(json);
    }
    /**
     *  获取拓扑图数据
     *
     * @return
     */
    @GetMapping("selectTopology")
    public String selectTopology(){
        Map<String,Object> map = manager.selectTopology();
        JSONObject json = JSONObject.fromObject(map);
        return json.toString();
    }

    /**
     *  获取拓扑图数据(从内存读取数据   假数据）
     *
     * @return
     */
    @GetMapping("selectTopology2")
    public String selectTopology2(){

        if (topologyConfig.getJson().size()==0){
//        String data = "{\"nodeDataArray\":[{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"签名验签接入服务\",\"pos\":\"-2483.6842129933902 14.593516452874084\",\"serverName\":\"签名验签接入服务\",\"tcpTransactionTotalCount\":0,\"text\":\"签名验签接入服务\",\"url\":\"http://192.168.65.4:8889/swtz/api/v1/sign/viid\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"核心调度服务\",\"pos\":\"-1797.265820568994 -266.9314959929493\",\"serverName\":\"核心调度服务\",\"tcpTransactionTotalCount\":0,\"text\":\"核心调度服务\",\"url\":\"http://10.104.64.10:8900/api/v1/ytcr-basic\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"算法融合调度\",\"pos\":\"-1098.6494613611756 -274.18377752850535\",\"serverName\":\"算法融合调度\",\"tcpTransactionTotalCount\":0,\"text\":\"算法融合调度\",\"url\":\"wu\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"核心业务区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"其他业务\",\"pos\":\"-1419.1931649568446 13.029493955336704\",\"serverName\":\"其他业务\",\"tcpTransactionTotalCount\":0,\"text\":\"其他业务\",\"url\":\"http://10.104.64.10:9999/inapiservice/specialData\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"业务统计服务\",\"pos\":\"-2475.798197313072 -414.1359515492054\",\"serverName\":\"业务统计服务\",\"tcpTransactionTotalCount\":0,\"text\":\"业务统计服务\",\"url\":\"http://192.168.65.27:8780/statistics/getVisits\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"账户管理服务\",\"pos\":\"-2477.2212996562757 -315.53916851220026\",\"serverName\":\"账户管理服务\",\"tcpTransactionTotalCount\":0,\"text\":\"账户管理服务\",\"url\":\"http://10.104.64.62:8764/swtz/api/v1/account/findAppData?pageNum=1&pageSize=1\"},{\"caption\":\"\",\"category\":\"serviceGroup\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入区\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"接入管理\",\"pos\":\"-2470.4441227074676 67.10778818688652\",\"serverName\":\"接入管理\",\"tcpTransactionTotalCount\":0,\"text\":\"接入管理\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"VPN方式接入服务\",\"pos\":\"-2480.765318757338 138.74851250653688\",\"serverName\":\"VPN方式接入服务\",\"tcpTransactionTotalCount\":0,\"text\":\"VPN方式接入服务\",\"url\":\"http://10.104.64.10:8888/swtz/api/v1/vpn/ytcr-basic\"},{\"caption\":\"\",\"category\":\"machine\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"mysql组件\",\"pos\":\"-2754.3462306600863 -331.02918266520214\",\"serverName\":\"mysql组件\",\"tcpTransactionTotalCount\":0,\"text\":\"mysql组件\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"machine\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"平台运营管理\",\"pos\":\"-2190.814288136425 -422.9124891980362\",\"serverName\":\"平台运营管理\",\"tcpTransactionTotalCount\":0,\"text\":\"平台运营管理\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"customer\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"公安网VPN用户\",\"pos\":\"-2958.8505203259415 123.2627988464879\",\"serverName\":\"公安网VPN用户\",\"tcpTransactionTotalCount\":0,\"text\":\"公安网VPN用户\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"token方式接入服务\",\"pos\":\"-2474.9840242230107 -90.28859707524175\",\"serverName\":\"token方式接入服务\",\"tcpTransactionTotalCount\":0,\"text\":\"token方式接入服务\",\"url\":\"http://10.104.64.10:8887/swtz/api/v1/token/ytcr-basic\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"算法区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"人脸识别引擎2\",\"pos\":\"-797.662751136269 -397.4312164130236\",\"serverName\":\"人脸识别引擎2\",\"tcpTransactionTotalCount\":0,\"text\":\"人脸识别引擎2\",\"url\":\"http://10.104.96.7:10006/algorithm/compareImageAndImage\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"算法区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"图像质量评估引擎\",\"pos\":\"-797.2511420874591 -123.30969487840144\",\"serverName\":\"图像质量评估引擎\",\"tcpTransactionTotalCount\":0,\"text\":\"图像质量评估引擎\",\"url\":\"http://10.104.96.7:10006/algorithm/getFeature\"},{\"caption\":\"\",\"category\":\"customer\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"公安网用户\",\"pos\":\"-2948.0090230389205 -55.12663614433889\",\"serverName\":\"公安网用户\",\"tcpTransactionTotalCount\":0,\"text\":\"公安网用户\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"serviceGroup\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入区\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"权限管理\",\"pos\":\"-2470.4441227074676 67.10778818688652\",\"serverName\":\"权限管理\",\"tcpTransactionTotalCount\":0,\"text\":\"权限管理\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"算法区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"其他引擎\",\"pos\":\"-787.8861025138158 32.03184687453745\",\"serverName\":\"其他引擎\",\"tcpTransactionTotalCount\":0,\"text\":\"其他引擎\",\"url\":\"http://10.104.96.7:10006/algorithm/compareImageAndFeature\"},{\"caption\":\"\",\"category\":\"serviceGroup\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"日志汇聚\",\"pos\":\"-2470.4441227074676 67.10778818688652\",\"serverName\":\"日志汇聚\",\"tcpTransactionTotalCount\":0,\"text\":\"日志汇聚\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"算法区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"人脸识别引擎1\",\"pos\":\"-799.0868618715551 -548.3701026800203\",\"serverName\":\"人脸识别引擎1\",\"tcpTransactionTotalCount\":0,\"text\":\"人脸识别引擎1\",\"url\":\"http://10.104.96.7:10006/algorithm/compareFeatureAndFeature\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"权限核验facl\",\"pos\":\"-2952.3097268671577 -392.4089634839355\",\"serverName\":\"权限核验facl\",\"tcpTransactionTotalCount\":0,\"text\":\"权限核验facl\",\"url\":\"http://10.104.64.10:8763/facl/api/v1/access/frifffcc207-3b64-4736-bbbf-614e85e5eb8c\"},{\"caption\":\"\",\"category\":\"machine\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"签名服务器\",\"pos\":\"-2810.2833206425184 47.4107677238989\",\"serverName\":\"签名服务器\",\"tcpTransactionTotalCount\":0,\"text\":\"签名服务器\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"serviceGroup\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"接入区\",\"pos\":\"-2470.4441227074676 67.10778818688652\",\"serverName\":\"接入区\",\"tcpTransactionTotalCount\":0,\"text\":\"接入区\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"核心业务区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"实时身份识别\",\"pos\":\"-1429.1764344884002 -547.4295876965392\",\"serverName\":\"实时身份识别\",\"tcpTransactionTotalCount\":0,\"text\":\"实时身份识别\",\"url\":\"http://10.104.64.10:9999/inapiservice/syncData\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"业务权限配置\",\"pos\":\"-2473.1502205925026 -509.51180663793565\",\"serverName\":\"业务权限配置\",\"tcpTransactionTotalCount\":0,\"text\":\"业务权限配置\",\"url\":\"http://10.104.64.61:8773/swtz/api/v1/permission/areas\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"核心业务区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"聚档人员实名化\",\"pos\":\"-1433.8018857081256 -267.70802709631556\",\"serverName\":\"聚档人员实名化\",\"tcpTransactionTotalCount\":0,\"text\":\"聚档人员实名化\",\"url\":\"http://10.104.64.10:9999/inapiservice/syncDataMCompareN\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"算法区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"人脸识别引擎3\",\"pos\":\"-795.6335828597649 -273.5497705833391\",\"serverName\":\"人脸识别引擎3\",\"tcpTransactionTotalCount\":0,\"text\":\"人脸识别引擎3\",\"url\":\"http://10.104.96.7:10006/algorithm/getTopNByImage\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"token核验服务\",\"pos\":\"-2805.959294573447 -116.62194783072228\",\"serverName\":\"token核验服务\",\"tcpTransactionTotalCount\":0,\"text\":\"token核验服务\",\"url\":\"http://10.104.64.10:8886/swtz/api/v1/appid/frifffcc207-3b64-4736-bbbf-614e85e5eb8c\"},{\"caption\":\"\",\"category\":\"serviceGroup\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"核心业务区\",\"pos\":\"-2470.4441227074676 67.10778818688652\",\"serverName\":\"核心业务区\",\"    tcpTransactionTotalCount\":0,\"text\":\"核心业务区\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"日志汇聚\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"日志汇聚服务\",\"pos\":\"-1801.991712515044 137.82286305583273\",\"serverName\":\"日志汇聚服务\",\"tcpTransactionTotalCount\":0,\"text\":\"日志汇聚服务\",\"url\":\"http://10.104.64.10:9998/swtz/api/v1/log/yxinfo\"},{\"caption\":\"\",\"category\":\"machine\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"redis组件\",\"pos\":\"-2752.3949616380514 -459.46971969996963\",\"serverName\":\"redis组件\",\"tcpTransactionTotalCount\":0,\"text\":\"redis组件\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"serviceGroup\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"算法区\",\"pos\":\"-2470.4441227074676 67.10778818688652\",\"serverName\":\"算法区\",\"tcpTransactionTotalCount\":0,\"text\":\"算法区\",\"url\":\"\"}],\"linkDataArray\":[{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"签名验签接入服务\",\"fromSpot\":\"left\",\"points\":[-2463.6843,30.988125,-2359.6614,-15.968698,-1923.9491,-194.54588,-1817.2659,-233.9528],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"核心调度服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"签名验签接入服务\",\"fromSpot\":\"left\",\"points\":[-2503.6843,40.9751,-2547.3394,42.90178,-2740.3103,55.27648,-2790.2832,59.64551],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"签名服务器\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"核心调度服务\",\"fromSpot\":\"left\",\"points\":[-1778.57,-221.56851,-1511.687,63.799915,-1443.9229,21.993011,-1439.1931,23.02593],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"其他业务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"核心调度服务\",\"fromSpot\":\"left\",\"points\":[-1777.2659,-257.08884,-1630.3828,-371.80756,-1460.2872,-504.0138,-1449.1764,-510.383],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"实时身份识别\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"核心调度服务\",\"fromSpot\":\"left\",\"points\":[-1777.2659,-241.82465,-1631.8348,-244.44495,-1463.7216,-246.96773,-1453.8019,-245.32689],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"聚档人员实名化\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"算法融合调度\",\"fromSpot\":\"left\",\"points\":[-1078.6494,-265.64542,-1005.8395,-327.86285,-893.29407,-430.25266,-819.08685,-503.2131],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"人脸识别引擎1\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"算法融合调度\",\"fromSpot\":\"left\",\"points\":[-1078.6494,-254.66212,-1034.5139,-268.36407,-858.54407,-340.22067,-817.6628,-361.5374],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"人脸识别引擎2\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"算法融合调度\",\"fromSpot\":\"left\",\"points\":[-1078.6494,-248.81067,-1032.9724,-248.65497,-865.7189,-240.44527,-815.6336,-246.1633],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"人脸识别引擎3\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"算法融合调度\",\"fromSpot\":\"left\",\"points\":[-1080.562,-228.82079,-1038.0873,-182.97966,-904.19714,31.189457,-807.8861,43.493504],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"其他引擎\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"算法融合调度\",\"fromSpot\":\"left\",\"points\":[-1078.6494,-236.42659,-1038.5028,-211.80801,-877.49536,-136.79707,-817.25116,-107.44187],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"图像质量评估引擎\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"其他业务\",\"fromSpot\":\"left\",\"points\":[-1399.1931,23.896997,-1328.0433,12.653117,-1164.8182,-192.40381,-1118.6494,-231.4268],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"算法融合调度\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"业务统计服务\",\"fromSpot\":\"left\",\"points\":[-2455.798,-387.69333,-2415.1838,-385.45975,-2244.9146,-397.69482,-2210.8142,-397.6732],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"平台运营管理\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"账户管理服务\",\"fromSpot\":\"left\",\"points\":[-2457.2212,-298.46118,-2417.9504,-314.527,-2258.0413,-372.87262,-2210.8142,-390.389],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"平台运营管理\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"接入管理\",\"fromSpot\":\"left\",\"points\":[-2869.6797,-204.374,-2896.1216,-238.69055,-2875.425,-258.3238,-2873.446,-259.17618],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"权限管理\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"VPN方式接入服务\",\"fromSpot\":\"left\",\"points\":[-2460.7654,153.17299,-2353.6616,94.28801,-1919.9215,-163.01274,-1817.2659,-228.56119],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"核心调度服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"mysql组件\",\"fromSpot\":\"left\",\"points\":[-2734.3464,-331.11685,-2731.0532,-333.06055,-2645.1074,-381.16785,-2493.1504,-472.2243],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"业务权限配置\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"mysql组件\",\"fromSpot\":\"left\",\"points\":[-2734.3464,-320.99142,-2664.795,-334.98105,-2555.1672,-367.38464,-2495.798,-383.1238],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"业务统计服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"mysql组件\",\"fromSpot\":\"left\",\"points\":[-2734.3464,-315.70746,-2690.367,-312.5191,-2547.7058,-294.00757,-2497.2214,-291.34348],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"账户管理服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"公安网VPN用户\",\"fromSpot\":\"left\",\"points\":[-2938.8506,150.44135,-2867.3403,156.73438,-2571.7617,166.20456,-2500.7654,164.78648],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"VPN方式接入服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"token方式接入服务\",\"fromSpot\":\"left\",\"points\":[-2454.9841,-70.880875,-2351.9187,-101.8834,-1922.0442,-209.32562,-1817.2659,-236.20508],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"核心调度服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"公安网用户\",\"fromSpot\":\"left\",\"points\":[-2928.009,-41.01175,-2857.0012,-40.59659,-2564.8538,-55.36602,-2494.984,-62.59373],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"token方式接入服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"公安网用户\",\"fromSpot\":\"left\",\"points\":[-2928.009,-36.38166,-2860.4438,-20.182405,-2546.3657,26.163107,-2503.6843,35.596886],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"签名验签接入服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"权限核验facl\",\"fromSpot\":\"left\",\"points\":[-2932.3098,-364.0611,-2914.8794,-361.198,-2824.3064,-338.75204,-2774.3464,-323.53052],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"mysql组件\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"权限核验facl\",\"fromSpot\":\"left\",\"points\":[-2932.3098,-375.40027,-2901.0332,-388.10587,-2806.254,-422.93652,-2772.3948,-437.50882],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"redis组件\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"实时身份识别\",\"fromSpot\":\"left\",\"points\":[-1409.1764,-506.32898,-1361.4991,-469.00958,-1162.8474,-304.7901,-1118.6494,-266.00214],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"算法融合调度\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"业务权限配置\",\"fromSpot\":\"left\",\"points\":[-2453.1501,-477.12677,-2412.5537,-463.0288,-2249.673,-411.5473,-2210.8142,-402.1905],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"平台运营管理\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"聚档人员实名化\",\"fromSpot\":\"left\",\"points\":[-1413.8019,-242.91452,-1376.2797,-244.25546,-1198.8707,-244.01991,-1118.6494,-247.67133],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"算法融合调度\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"token核验服务\",\"fromSpot\":\"left\",\"points\":[-2785.9592,-89.94587,-2655.0508,-81.62456,-2503.7654,-71.50079,-2494.984,-69.11964],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"token方式接入服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"日志汇聚服务\",\"fromSpot\":\"\",\"points\":[-1821.9917,167.86627,-2425.7725,309.14423,-2524.9207,237.56937,-2528.9624,201.1115],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"接入区\",\"toSpot\":\"\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"日志汇聚服务\",\"fromSpot\":\"\",\"points\":[-1781.9917,159.84584,-1417.4015,98.852615,-1441.3502,71.50415,-1440.8514,57.960987],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"核心业务区\",\"toSpot\":\"\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"日志汇聚服务\",\"fromSpot\":\"\",\"points\":[-1781.9917,162.91415,-1394.6392,158.37663,-816.9019,278.02087,-810.25726,76.96334],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"算法区\",\"toSpot\":\"\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"redis组件\",\"fromSpot\":\"left\",\"points\":[-2732.3948,-451.0714,-2715.0898,-455.6468,-2626.9282,-465.72855,-2493.1504,-481.8344],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"token核验业务权限配置服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"redis组件\",\"fromSpot\":\"left\",\"points\":[-2732.3948,-451.0714,-2715.0898,-455.6468,-2626.9282,-465.72855,-2493.1504,-481.8344],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"业务权限配置\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"redis组件\",\"fromSpot\":\"left\",\"points\":[-2732.3948,-439.11536,-2694.3455,-427.1653,-2533.9192,-394.0032,-2495.798,-390.19513],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"业务统计服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"redis组件\",\"fromSpot\":\"left\",\"points\":[-2732.3948,-434.45917,-2698.6963,-415.63425,-2531.7256,-324.80646,-2497.2214,-303.1389],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"账户管理服务\",\"toSpot\":\"right\"}]}";
            String data = "{\"nodeDataArray\":[{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"签名验签接入服务\",\"pos\":\"-2483.6842129933902 14.593516452874084\",\"serverName\":\"签名验签接入服务\",\"tcpTransactionTotalCount\":0,\"text\":\"签名验签接入服务\",\"url\":\"http://192.168.65.4:8889/swtz/api/v1/sign/viid\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"核心调度服务\",\"pos\":\"-1797.265820568994 -266.9314959929493\",\"serverName\":\"核心调度服务\",\"tcpTransactionTotalCount\":0,\"text\":\"核心调度服务\",\"url\":\"http://10.104.64.10:8900/api/v1/ytcr-basic\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"算法融合调度\",\"pos\":\"-1098.6494613611756 -274.18377752850535\",\"serverName\":\"算法融合调度\",\"tcpTransactionTotalCount\":0,\"text\":\"算法融合调度\",\"url\":\"wu\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"核心业务区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"其他业务\",\"pos\":\"-1419.1931649568446 13.029493955336704\",\"serverName\":\"其他业务\",\"tcpTransactionTotalCount\":0,\"text\":\"其他业务\",\"url\":\"http://10.104.64.10:9999/inapiservice/specialData\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"业务统计服务\",\"pos\":\"-2475.798197313072 -414.1359515492054\",\"serverName\":\"业务统计服务\",\"tcpTransactionTotalCount\":0,\"text\":\"业务统计服务\",\"url\":\"http://192.168.65.27:8780/statistics/getVisits\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"账户管理服务\",\"pos\":\"-2477.2212996562757 -315.53916851220026\",\"serverName\":\"账户管理服务\",\"tcpTransactionTotalCount\":0,\"text\":\"账户管理服务\",\"url\":\"http://10.104.64.62:8764/swtz/api/v1/account/findAppData?pageNum=1&pageSize=1\"},{\"caption\":\"\",\"category\":\"serviceGroup\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入区\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"接入管理\",\"pos\":\"-2470.4441227074676 67.10778818688652\",\"serverName\":\"接入管理\",\"tcpTransactionTotalCount\":0,\"text\":\"接入管理\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"VPN方式接入服务\",\"pos\":\"-2480.765318757338 138.74851250653688\",\"serverName\":\"VPN方式接入服务\",\"tcpTransactionTotalCount\":0,\"text\":\"VPN方式接入服务\",\"url\":\"http://10.104.64.10:8888/swtz/api/v1/vpn/ytcr-basic\"},{\"caption\":\"\",\"category\":\"machine\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"mysql组件\",\"pos\":\"-2754.3462306600863 -331.02918266520214\",\"serverName\":\"mysql组件\",\"tcpTransactionTotalCount\":0,\"text\":\"mysql组件\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"machine\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"平台运营管理\",\"pos\":\"-2190.814288136425 -422.9124891980362\",\"serverName\":\"平台运营管理\",\"tcpTransactionTotalCount\":0,\"text\":\"平台运营管理\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"customer\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"公安网VPN用户\",\"pos\":\"-2958.8505203259415 123.2627988464879\",\"serverName\":\"公安网VPN用户\",\"tcpTransactionTotalCount\":0,\"text\":\"公安网VPN用户\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"token方式接入服务\",\"pos\":\"-2474.9840242230107 -90.28859707524175\",\"serverName\":\"token方式接入服务\",\"tcpTransactionTotalCount\":0,\"text\":\"token方式接入服务\",\"url\":\"http://10.104.64.10:8887/swtz/api/v1/token/ytcr-basic\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"算法区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"人脸识别引擎2\",\"pos\":\"-797.662751136269 -397.4312164130236\",\"serverName\":\"人脸识别引擎2\",\"tcpTransactionTotalCount\":0,\"text\":\"人脸识别引擎2\",\"url\":\"http://10.104.96.7:10006/algorithm/compareImageAndImage\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"算法区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"图像质量评估引擎\",\"pos\":\"-797.2511420874591 -123.30969487840144\",\"serverName\":\"图像质量评估引擎\",\"tcpTransactionTotalCount\":0,\"text\":\"图像质量评估引擎\",\"url\":\"http://10.104.96.7:10006/algorithm/getFeature\"},{\"caption\":\"\",\"category\":\"customer\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"公安网用户\",\"pos\":\"-2948.0090230389205 -55.12663614433889\",\"serverName\":\"公安网用户\",\"tcpTransactionTotalCount\":0,\"text\":\"公安网用户\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"serviceGroup\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入区\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"权限管理\",\"pos\":\"-2470.4441227074676 67.10778818688652\",\"serverName\":\"权限管理\",\"tcpTransactionTotalCount\":0,\"text\":\"权限管理\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"算法区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"其他引擎\",\"pos\":\"-787.8861025138158 32.03184687453745\",\"serverName\":\"其他引擎\",\"tcpTransactionTotalCount\":0,\"text\":\"其他引擎\",\"url\":\"http://10.104.96.7:10006/algorithm/compareImageAndFeature\"},{\"caption\":\"\",\"category\":\"serviceGroup\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"日志汇聚\",\"pos\":\"-2470.4441227074676 67.10778818688652\",\"serverName\":\"日志汇聚\",\"tcpTransactionTotalCount\":0,\"text\":\"日志汇聚\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"算法区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"人脸识别引擎1\",\"pos\":\"-799.0868618715551 -548.3701026800203\",\"serverName\":\"人脸识别引擎1\",\"tcpTransactionTotalCount\":0,\"text\":\"人脸识别引擎1\",\"url\":\"http://10.104.96.7:10006/algorithm/compareFeatureAndFeature\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"权限核验facl\",\"pos\":\"-2952.3097268671577 -392.4089634839355\",\"serverName\":\"权限核验facl\",\"tcpTransactionTotalCount\":0,\"text\":\"权限核验facl\",\"url\":\"http://10.104.64.10:8763/facl/api/v1/access/frifffcc207-3b64-4736-bbbf-614e85e5eb8c\"},{\"caption\":\"\",\"category\":\"machine\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"签名服务器\",\"pos\":\"-2810.2833206425184 47.4107677238989\",\"serverName\":\"签名服务器\",\"tcpTransactionTotalCount\":0,\"text\":\"签名服务器\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"serviceGroup\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"接入区\",\"pos\":\"-2470.4441227074676 67.10778818688652\",\"serverName\":\"接入区\",\"tcpTransactionTotalCount\":0,\"text\":\"接入区\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"核心业务区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"实时身份识别\",\"pos\":\"-1429.1764344884002 -547.4295876965392\",\"serverName\":\"实时身份识别\",\"tcpTransactionTotalCount\":0,\"text\":\"实时身份识别\",\"url\":\"http://10.104.64.10:9999/inapiservice/syncData\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"业务权限配置\",\"pos\":\"-2473.1502205925026 -509.51180663793565\",\"serverName\":\"业务权限配置\",\"tcpTransactionTotalCount\":0,\"text\":\"业务权限配置\",\"url\":\"http://10.104.64.61:8773/swtz/api/v1/permission/areas\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"核心业务区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"聚档人员实名化\",\"pos\":\"-1433.8018857081256 -267.70802709631556\",\"serverName\":\"聚档人员实名化\",\"tcpTransactionTotalCount\":0,\"text\":\"聚档人员实名化\",\"url\":\"http://10.104.64.10:9999/inapiservice/syncDataMCompareN\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"算法区\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"人脸识别引擎3\",\"pos\":\"-795.6335828597649 -273.5497705833391\",\"serverName\":\"人脸识别引擎3\",\"tcpTransactionTotalCount\":0,\"text\":\"人脸识别引擎3\",\"url\":\"http://10.104.96.7:10006/algorithm/getTopNByImage\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"接入管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"token核验服务\",\"pos\":\"-2805.959294573447 -116.62194783072228\",\"serverName\":\"token核验服务\",\"tcpTransactionTotalCount\":0,\"text\":\"token核验服务\",\"url\":\"http://10.104.64.10:8886/swtz/api/v1/appid/frifffcc207-3b64-4736-bbbf-614e85e5eb8c\"},{\"caption\":\"\",\"category\":\"serviceGroup\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"核心业务区\",\"pos\":\"-2470.4441227074676 67.10778818688652\",\"serverName\":\"核心业务区\",\"    tcpTransactionTotalCount\":0,\"text\":\"核心业务区\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"日志汇聚\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"日志汇聚服务\",\"pos\":\"-1801.991712515044 137.82286305583273\",\"serverName\":\"日志汇聚服务\",\"tcpTransactionTotalCount\":0,\"text\":\"日志汇聚服务\",\"url\":\"http://10.104.64.10:9998/swtz/api/v1/log/yxinfo\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"日志汇聚\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\".\",\"pos\":\"-1801.991712515044 137.82286305583273\",\"serverName\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"url\":\"http://10.104.64.10:9998/swtz/api/v1/log/yxinfo\"},{\"caption\":\"\",\"category\":\"clusterNormal\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"日志汇聚\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"..\",\"pos\":\"-1801.991712515044 137.82286305583273\",\"serverName\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"url\":\"http://10.104.64.10:9998/swtz/api/v1/log/yxinfo\"},{\"caption\":\"\",\"category\":\"machine\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"权限管理\",\"imgsrc\":\"\",\"isGroup\":false,\"key\":\"redis组件\",\"pos\":\"-2752.3949616380514 -459.46971969996963\",\"serverName\":\"redis组件\",\"tcpTransactionTotalCount\":0,\"text\":\"redis组件\",\"url\":\"\"},{\"caption\":\"\",\"category\":\"serviceGroup\",\"description\":\"服务异常数：0/0\\r\\n\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"faultCount\":0,\"group\":\"\",\"imgsrc\":\"\",\"isGroup\":true,\"key\":\"算法区\",\"pos\":\"-2470.4441227074676 67.10778818688652\",\"serverName\":\"算法区\",\"tcpTransactionTotalCount\":0,\"text\":\"算法区\",\"url\":\"\"}],\"linkDataArray\":[{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"签名验签接入服务\",\"fromSpot\":\"left\",\"points\":[-2463.6843,30.988125,-2359.6614,-15.968698,-1923.9491,-194.54588,-1817.2659,-233.9528],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"核心调度服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"签名验签接入服务\",\"fromSpot\":\"left\",\"points\":[-2503.6843,40.9751,-2547.3394,42.90178,-2740.3103,55.27648,-2790.2832,59.64551],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"签名服务器\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"核心调度服务\",\"fromSpot\":\"left\",\"points\":[-1778.57,-221.56851,-1511.687,63.799915,-1443.9229,21.993011,-1439.1931,23.02593],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"其他业务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"核心调度服务\",\"fromSpot\":\"left\",\"points\":[-1777.2659,-257.08884,-1630.3828,-371.80756,-1460.2872,-504.0138,-1449.1764,-510.383],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"实时身份识别\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"核心调度服务\",\"fromSpot\":\"left\",\"points\":[-1777.2659,-241.82465,-1631.8348,-244.44495,-1463.7216,-246.96773,-1453.8019,-245.32689],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"聚档人员实名化\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"算法融合调度\",\"fromSpot\":\"left\",\"points\":[-1078.6494,-265.64542,-1005.8395,-327.86285,-893.29407,-430.25266,-819.08685,-503.2131],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"人脸识别引擎1\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"算法融合调度\",\"fromSpot\":\"left\",\"points\":[-1078.6494,-254.66212,-1034.5139,-268.36407,-858.54407,-340.22067,-817.6628,-361.5374],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"人脸识别引擎2\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"算法融合调度\",\"fromSpot\":\"left\",\"points\":[-1078.6494,-248.81067,-1032.9724,-248.65497,-865.7189,-240.44527,-815.6336,-246.1633],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"人脸识别引擎3\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"算法融合调度\",\"fromSpot\":\"left\",\"points\":[-1080.562,-228.82079,-1038.0873,-182.97966,-904.19714,31.189457,-807.8861,43.493504],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"其他引擎\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"算法融合调度\",\"fromSpot\":\"left\",\"points\":[-1078.6494,-236.42659,-1038.5028,-211.80801,-877.49536,-136.79707,-817.25116,-107.44187],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"图像质量评估引擎\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"其他业务\",\"fromSpot\":\"left\",\"points\":[-1399.1931,23.896997,-1328.0433,12.653117,-1164.8182,-192.40381,-1118.6494,-231.4268],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"算法融合调度\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"业务统计服务\",\"fromSpot\":\"left\",\"points\":[-2455.798,-387.69333,-2415.1838,-385.45975,-2244.9146,-397.69482,-2210.8142,-397.6732],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"平台运营管理\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"账户管理服务\",\"fromSpot\":\"left\",\"points\":[-2457.2212,-298.46118,-2417.9504,-314.527,-2258.0413,-372.87262,-2210.8142,-390.389],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"平台运营管理\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"接入管理\",\"fromSpot\":\"left\",\"points\":[-2869.6797,-204.374,-2896.1216,-238.69055,-2875.425,-258.3238,-2873.446,-259.17618],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"权限管理\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"VPN方式接入服务\",\"fromSpot\":\"left\",\"points\":[-2460.7654,153.17299,-2353.6616,94.28801,-1919.9215,-163.01274,-1817.2659,-228.56119],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"核心调度服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"mysql组件\",\"fromSpot\":\"left\",\"points\":[-2734.3464,-331.11685,-2731.0532,-333.06055,-2645.1074,-381.16785,-2493.1504,-472.2243],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"业务权限配置\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"mysql组件\",\"fromSpot\":\"left\",\"points\":[-2734.3464,-320.99142,-2664.795,-334.98105,-2555.1672,-367.38464,-2495.798,-383.1238],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"业务统计服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"mysql组件\",\"fromSpot\":\"left\",\"points\":[-2734.3464,-315.70746,-2690.367,-312.5191,-2547.7058,-294.00757,-2497.2214,-291.34348],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"账户管理服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"公安网VPN用户\",\"fromSpot\":\"left\",\"points\":[-2938.8506,150.44135,-2867.3403,156.73438,-2571.7617,166.20456,-2500.7654,164.78648],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"VPN方式接入服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"token方式接入服务\",\"fromSpot\":\"left\",\"points\":[-2454.9841,-70.880875,-2351.9187,-101.8834,-1922.0442,-209.32562,-1817.2659,-236.20508],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"核心调度服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"公安网用户\",\"fromSpot\":\"left\",\"points\":[-2928.009,-41.01175,-2857.0012,-40.59659,-2564.8538,-55.36602,-2494.984,-62.59373],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"token方式接入服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"公安网用户\",\"fromSpot\":\"left\",\"points\":[-2928.009,-36.38166,-2860.4438,-20.182405,-2546.3657,26.163107,-2503.6843,35.596886],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"签名验签接入服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"权限核验facl\",\"fromSpot\":\"left\",\"points\":[-2932.3098,-364.0611,-2914.8794,-361.198,-2824.3064,-338.75204,-2774.3464,-323.53052],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"mysql组件\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"权限核验facl\",\"fromSpot\":\"left\",\"points\":[-2932.3098,-375.40027,-2901.0332,-388.10587,-2806.254,-422.93652,-2772.3948,-437.50882],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"redis组件\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"实时身份识别\",\"fromSpot\":\"left\",\"points\":[-1409.1764,-506.32898,-1361.4991,-469.00958,-1162.8474,-304.7901,-1118.6494,-266.00214],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"算法融合调度\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"业务权限配置\",\"fromSpot\":\"left\",\"points\":[-2453.1501,-477.12677,-2412.5537,-463.0288,-2249.673,-411.5473,-2210.8142,-402.1905],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"平台运营管理\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"聚档人员实名化\",\"fromSpot\":\"left\",\"points\":[-1413.8019,-242.91452,-1376.2797,-244.25546,-1198.8707,-244.01991,-1118.6494,-247.67133],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"算法融合调度\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"token核验服务\",\"fromSpot\":\"left\",\"points\":[-2785.9592,-89.94587,-2655.0508,-81.62456,-2503.7654,-71.50079,-2494.984,-69.11964],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"token方式接入服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"日志汇聚\",\"fromSpot\":\"\",\"points\":[-1821.9917,167.86627,-2425.7725,309.14423,-2524.9207,237.56937,-2528.9624,201.1115],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"接入区\",\"toSpot\":\"\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"日志汇聚\",\"fromSpot\":\"\",\"points\":[-1781.9917,159.84584,-1417.4015,98.852615,-1441.3502,71.50415,-1440.8514,57.960987],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"核心业务区\",\"toSpot\":\"\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"日志汇聚\",\"fromSpot\":\"\",\"points\":[-1781.9917,162.91415,-1394.6392,158.37663,-816.9019,278.02087,-810.25726,76.96334],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"算法区\",\"toSpot\":\"\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"redis组件\",\"fromSpot\":\"left\",\"points\":[-2732.3948,-451.0714,-2715.0898,-455.6468,-2626.9282,-465.72855,-2493.1504,-481.8344],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"业务权限配置\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"redis组件\",\"fromSpot\":\"left\",\"points\":[-2732.3948,-439.11536,-2694.3455,-427.1653,-2533.9192,-394.0032,-2495.798,-390.19513],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"业务统计服务\",\"toSpot\":\"right\"},{\"client_alias\":\"\",\"color\":\"#008000\",\"description\":\"\",\"descriptionCss\":\"\",\"errorTcpTransactionTotalCount\":0,\"from\":\"redis组件\",\"fromSpot\":\"left\",\"points\":[-2732.3948,-434.45917,-2698.6963,-415.63425,-2531.7256,-324.80646,-2497.2214,-303.1389],\"server_alias\":\"\",\"tcpTransactionTotalCount\":0,\"text\":\"\",\"to\":\"账户管理服务\",\"toSpot\":\"right\"}]}";
            JSONObject jsonObject = JSONObject.fromObject(data);
            topologyConfig.getJson().put("linkDataArray",jsonObject.getJSONArray("linkDataArray"));
            topologyConfig.getJson().put("nodeDataArray",jsonObject.getJSONArray("nodeDataArray"));
        }
        return topologyConfig.getJson().toString();
    }

    /**
     *  保存数据
     * @return
     */
    @PostMapping("saveTopology")
    public Map saveTopology(@RequestBody String json) {
        return manager.saveTopology(json);
    }
    /**
     *  获取认证量统计
     * @return
     */
    @GetMapping("getStatistics")
    public JSONObject getStatistics() {
        return manager.getStatistics();
    }

    /*
    * 微信告警开关
    * */
    @PostMapping("changeSwitch")
    public Map changeSwitch(@RequestBody String json) {
        return manager.changeSwitch(json);
    }
    /*
    * 获取微信告警开关状态
    * */
    @GetMapping("getSwitch")
    public Map getSwitch() {
        return manager.getSwitch();
    }

    //新建（Server_host_info）监控服务信息
    @PostMapping("insertServerHostInfo")
    public String insertServerHostInfo(@RequestBody String json){
        return manager.insertServerHostInfo(json);
    }
    //删除（Server_host_info）监控服务信息
    @PostMapping("delServerHostInfo")
    public String delServerHostInfo(@RequestBody String json){
        return manager.delServerHostInfo(json);
    }

    //修改（Server_host_info）监控服务信息
    @PostMapping("updateServerHostInfo")
    public String updateServerHostInfo(@RequestBody String json){
        return manager.updateServerHostInfo(json);
    }

}
